function time = str2time ( str )
%STR2TIME	Convert string to time vector.
%	TIME = STR2TIME(STR) converts the time in the string STR to
%	a time vector. Only 1 row of STR is used.
%	The output TIME structure is a vector with values as follows:
%	[ yyyy, mm, dd, hh, mm, ss.sss ].
%	Each element of TIME is verified to be within legal limits.
%
%	Several input formats are recognized:
%	  dd-mmm-yy hh:mm:ss.sss
%	  yyyy/mm/dd hh:mm:ss:sss
%	  yy/mm/dd hh:mm:ss.sss
%	  yeardoy hh:mm:ss.sss
%	  epochal
%	Up to 6 input parameters are used as appropriate. Date and time 
%	parameters may be separated by any character.
%	If no numbers are read from STR, or no STR is given, current clock 
%	time is returned.
%
%	See also TIME2STR, STR2DUR.

%	MatSeis 1.6
%	Mark Harris, mharris@sandia.gov
%	Copyright (c) 1996-2001 Sandia National Laboratories. All rights reserved.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Default output.
%
time = [ 1970 1 1 0 0 0 ];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Check arguments.
%
if ( nargin < 1 )
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % Current time requested.
  %
  time = clock;
  return;
end
if ( isempty ( sscanf ( str, '%d' ) ) )
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % Current time requested.
  %
  time = clock;
  return;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Check for DATE format (dd-mmm-yy).
% To be DATE format, str must have at least 3 alphabet letters and they must 
% match a month abbreviation.
%
m = [];
letter = isletter ( str );
if ( sum(letter) >= 3 )
  month = upper(str(letter));
  months = { 'JAN'; 'FEB'; 'MAR'; 'APR'; 'MAY'; 'JUN'; ...
             'JUL'; 'AUG'; 'SEP'; 'OCT'; 'NOV'; 'DEC' };

  m = find( strcmp( month, months ) );
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Decode DATE string.
%
if ( ~ isempty ( m ) )
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % Read date string.
  %
  [t1,count] = sscanf ( str, '%d%*5c%d %d%*c%d%*c%f', 5 );

  if ( count < 2 )
    return;
  end

  t = [ t1(2) m t1(1) ];
  if ( count > 2 )
    t = [ t t1(3) ];
  end
  if ( count > 3 )
    t = [ t t1(4) ];
  end
  if ( count > 4 )
    t = [ t t1(5) ];
  end
  count = count + 1;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Decode TIME string.
%
else
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % Read time string.
  %
  [t,count] = sscanf ( str, '%d%*c%d%*c%d %d%*c%d%*c%f', 6 );
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Year
%
if ( t(1) >= 0 & t(1) < 20 )
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % Partial year entered (20XX)
  %
  time(1) = t(1) + 2000;
elseif ( t(1) >= 20 & t(1) < 100 )
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % Partial year entered (19XX)
  %
  time(1) = t(1) + 1900;
elseif ( t(1) >= 100 & t(1) < 3000 )
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % Full year entered
  %
  time(1) = t(1);
elseif ( t(1) >= 1000001 & t(1) < 2999367 )
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % Year-DOY entered
  %
  [time(1),time(2),time(3)] = ymd ( t(1) );
  t = [ 0; 0; t ];
  t(2) = time(2);
  t(3) = time(3);
  count = count+2;
else
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % Epochal time entered
  %
  time = etoh ( t(1) );
  return;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Month
%
if ( count > 1 )
  if ( t(2) < 1 | t(2) >= 13 )
    time(2) = 1;
    disp ( 'BAD TIME: invalid month' );
    return;
  else
    time(2) = t(2);
  end
else
  return;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Day
%
if ( count > 2 )
  if ( t(3) < 1 | t(3) >= 32 )
    time(3) = 1;
    disp ( 'BAD TIME: invalid day' );
    return;
  else
    time(3) = t(3);
  end
else
  return;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Hour
%
if ( count > 3 )
  if ( t(4) < 0 | t(4) >= 24 )
    time(4) = 0;
    disp ( 'BAD TIME: invalid hour' );
    return;
  else
    time(4) = t(4);
  end
else
  return;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Minute
%
if ( count > 4 )
  if ( t(5) < 0 | t(5) >= 60 )
    time(5) = 0;
    disp ( 'BAD TIME: invalid minute' );
    return;
  else
    time(5) = t(5);
  end
else
  return;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Second
%
if ( count > 5 )
  if ( t(6) < 0 | t(6) >= 60 )
    time(6) = 0;
    disp ( 'BAD TIME: invalid second' );
    return;
  else
    time(6) = t(6);
  end
else
  return;
end

