function sacdata_out=BCseis_process(sacdata_in,params,Wx_tem,xv1,yv1)
% BCseis_process - perform processing steps on data as determined by
% initial analysis with the BCseis GUI
% version 1.2, March 1, 2021

% Input:
%       sacdata_in - waveform in SAC format
%       params - data structure containing processing information
%       Wx_tem - CWT of a picked block
%       xv1, yv1 - polygon vertices of the picked block
%
% Output:
%       sacdata_out - processed waveform in SAC format

%****** params data structure *********************************************
% params.wavelet_type = "morlet" / "shannon" / "mhat" / "hhat"
% params.nvoices      = choice of (4 8 16 32)
% params.outfilesuf   = suffix that will be appended to end of filename for
%                        processed signal.
% params.subdir       = name of subdirectory for resulting captured signal
%                        if blank (''), then the file is written in the
%                        same directory as the input data.
%
% params.nbpblck   = 0, no band pass/block applied
%                  = 1 band block applied
%                  = -1, band pass applied (everything else zeroed out)
% params.scale_min = minimum scale for bandpass block/pass
% params.scale_max = maximum scale for bandpass block/pass
% params.bthresh   = percent amplitude for the block
%
% params.nnbnd = 0, no noise model considered
%              = 1, compute a noise model
% params.tstrn = noise start time in seismogram
% params.tfinn = noise finish time in seismogram
%
% params.noisethresh = 0, do not apply non-linear threshold to the noise
%                    = 1, apply hard threshold
%                    = 2, apply soft threshold
%
% params.signalthresh = 0, do not apply non-linear threshold to the signal
%                     = 1, apply hard threshold
%                     = 2, apply soft threshold
%
% params.nsig = number of standard deviations for the block threshold
%             = -1, apply Donoho's Threshold criterion
%             = -2, apply Empiridal CDF method to determine threshold
%                  (recommended)
%
% params.nsnr   = 0, do not apply the SNR detection method;
%               = 1, apply.  If applied, it will be completed before hard
%                  thresholding.
% params.nsnrlb = percent lower bound for SNR detection method
%
% params.nextract    = 0, do not extract a CWT block from a signal
%                    = 1, extract a CWT block from an array signal
%                    = 2, delete a CWT block from an array signal
% params.timetol      = time tolerance wanted for shifting the reference
%                      signal relative to the signal. This speeds up the
%                      correlation computation for windowing a block out of
%                      the CWT.  Shifts will occur for +/- params.timtol
% params.extpolyfile = filename for TFR polygon for the reference signal
% params.exttfrfile  = filename for the TFR of the reference signal
% 
%**************************************************************************
% compute the CWT
fprintf('Computing the CWT\n');
sacdata_new=sacdata_in;
sacdata_out=sacdata_new;

delta=1.0./sacdata_new(1).samprate;
npts=sacdata_new(1).nsamps;
x=sacdata_new(1).data(:,1);
t=linspace(0,delta.*(npts-1),npts);

wavelet_type=params.wavelet_type;
nvoices=params.nvoices;

[Wx_new,as_new] = cwt_fw(x,wavelet_type,nvoices,delta);

[na,n] = size(Wx_new);

%**************************************************************************
% Apply a bandpass block or pass, if wanted
if params.nbpblck == 1
    fprintf('Apply Band Block\n');
        %  Gather parameters
    scale_min=params.scale_min;
    scale_max=params.scale_max;
    thresh=params.bthresh.*0.01;
    
    %  save old CWT
    Wx_old=Wx_new;
    as_old=as_new;
    
    %  Threshold Wx_old to get Wx_new
    a(1,1:na)=1.0;
    a=a.*(as_old <= scale_min | as_old >= scale_max);
    for k=1:na
        if a(k) == 0
            a(k)=thresh;
        end
    end
    Wx_new=Wx_old.*a';
end

if params.nbpblck == -1
    fprintf('Apply Band Pass\n');
        %  Gather parameters
    scale_min=params.scale_min;
    scale_max=params.scale_max;
    thresh=params.bthresh.*0.01;
    
    %  save old CWT
    Wx_old=Wx_new;
    as_old=as_new;
    
    %  Threshold Wx_old to get Wx_new
    a(1,1:na)=1.0;
    a=a.*(as_old >= scale_min & as_old <= scale_max);
    for k=1:na
        if a(k) == 1
            a(k)=thresh;
        end
    end
    Wx_new=Wx_old.*a';
end
%**************************************************************************
% Calculate noise model and threshold function, if needed
if (params.nnbnd == 1 || params.noisethresh > 0 || params.signalthresh > 0)
%     fprintf('Calculate noise model and threshold function\n');
    % Get the time window

    begtime=params.tstrn;
    endtime=params.tfinn;

    nbeg=round(begtime./delta) + 1;
    nend=round(endtime./delta) + 1;
    
    if params.nsig == -1
        % compute Donoho's Threshold Criterion
        params.nsig=sqrt(2.*log10(nend-nbeg +1));
    end

    if params.nsig > -2
        % Assume Gaussian statistics
        % Calculate the mean and standard deviations at each scale in the
        % time-scale window
        M=mean(abs(Wx_new(:,nbeg:nend)'));
        S=std(abs(Wx_new(:,nbeg:nend)'));
        P=M + params.nsig.*S;
    else
        % Estimate empirical CDF for each scale, calculate 99% confidence 
        % level for the noise threshold
        [nrow,ncol]=size(Wx_new);
        conf=0.99;
        n_noise=nend-nbeg+1;
        for k=1:nrow
            W(1:n_noise)=abs(Wx_new(k,nbeg:nend))';
            [f,x]=ecdf(W);
            P(k)=interp1(f,x,conf);
        end
    end

else
end

%**************************************************************************
% apply the SNR detection method if wanted
if (params.nnbnd == 1 && params.nsnr == 1)
    
%     fprintf('Apply SNR detection method\n');
    
    nlbound=params.nsnrlb.*0.01;
    M_max=max(abs(M));
    Wx_new=Wx_new./(M'+nlbound.*M_max);
    
    % recalculate the noise model for possible further use
    M=mean(abs(Wx_new(:,nbeg:nend)'));
    S=std(abs(Wx_new(:,nbeg:nend)'));
else
end

%**************************************************************************
% Apply hard thresholding to the noise, if wanted (removing noise)
if params.noisethresh == 1
%     fprintf('Apply Hard Thresholding to noise\n');
    Wx_old=Wx_new;
    W_test=abs(Wx_old);
    Wx_new=Wx_old.*(P' < W_test);
else
end

%**************************************************************************
% Apply soft thresholding to the noise, if wanted (removing noise)
if params.noisethresh == 2
%     fprintf('Apply soft thresholding to noise\n');
    Wx_old=Wx_new;
    W_test=abs(Wx_old);
    Wx_new=sign(Wx_old).*(W_test - P').*(P' < W_test);
else
end

%**************************************************************************
% Apply hard thresholding to the signal, if wanted (removing signal)
if params.signalthresh == 1
%     fprintf('Apply hard thresholding to signal\n');
    Wx_old=Wx_new;
    W_test=abs(Wx_old);
    Wx_new=Wx_old.*(P' > W_test);
else
end

%**************************************************************************
% Apply soft thresholding to the signal, if wanted (removing signal)
if params.signalthresh == 2
%     fprintf('Apply soft thresholding to signal\n');
    Wx_old=Wx_new;
    W_test=abs(Wx_old);
    Wx_new=sign(Wx_old).* P'.*(P' <= W_test) + W_test.*(P' > W_test);
else
end

%**************************************************************************
% Apply Array Element Block Extraction or rejection, if wanted
if params.nextract > 0
    
%     fprintf('Apply array element block extraction or rejection\n');
    % Wx_tem,xv,yv are the input complex wavelet coefficients and polygon
    % vertices
    xv=xv1;
    yv=yv1;
    
    % find scale bounds of polygon
    [as_min]=min(10.^yv);
    [as_max]=max(10.^yv);
    
    nrows=length(as_new);
    for k=1:nrows-1
        if as_min >= as_new(k) && as_min < as_new(k+1); kmin=k; end
        if as_max > as_new(k) && as_max <= as_new(k+1); kmax=k+1; end
    end
    
    fprintf('as_min %g kmin %i as_max %g kmax %i\n',as_min,kmin,as_max,kmax);
    
    % Data CWT
    Wd(1:kmax-kmin+1,:)=abs(Wx_new(kmin:kmax,:));
    Wd=Wd.^3;
    [nrows,ncols]=size(Wd);
%     fprintf('nrows, ncols= %i %i\n',nrows,ncols);
    
    % Template CWT
    Wt(1:kmax-kmin+1,:)=abs(Wx_tem(kmin:kmax,:));
    Wt=Wt.^3;
    % calculate negative and positive time shifts that are allowed by
    % params.timetol
    nshifts=round(params.timetol./delta) + 1;
    
    klag=0;
    tsh(1:(2*nshifts +1))=0.0;
    Corr(1:(2*nshifts +1))=0.0;
    for nlag=-nshifts:1:nshifts
%         fprintf('nlag= %i\n',nlag);
        Wt_shift=circshift(Wt,nlag,2);
        Wprd=Wt_shift.*Wd;
        klag=klag+1;
        Corr(klag)=sum(sum(Wprd));
%         fprintf('klag= %i, Corr(klag)= %g\n',klag,Corr(klag));
    end
    
    % Calculate time shift for the polygon
    [Corr_max,kshift]=max(Corr);
    tshift=delta.*(kshift-nshifts - 1);
    
    xv=xv+tshift;
    
    % setup to find the polygon mask
    tstart=sacdata_in.headerA(6);
    t=linspace(tstart,tstart + delta.*(npts-1),npts);
    Wx_old=Wx_new;
    as_old=as_new;
    
    % WT is a na x n matrix
    % The matrix contains ones within the polygon, if this is the signal 
    %    that is wanted (params.nextract =1)
    % It contains zeros within the polygon and ones outside if this portion
    %    of the signal is not wanted (params.nextract = 2)
%     fprintf('Constructing Mask\n');
    if params.nextract == 1; WT=poly_mask_ones(t,as_old,xv,yv);end
    if params.nextract == 2; WT=poly_mask_zeros(t,as_old,xv,yv);end
    
    % scale-time block gate or barrier
    Wx_new=Wx_old.*WT;
  
else
    fprintf('No template available to correlate\n');
end
    

    
%**************************************************************************
% Compute the inverse CWT
fprintf('Computing inverse CWT\n');
anew=cwt_iw(Wx_new,wavelet_type,nvoices);
sacdata_out(1).data(1:npts,1)=anew(1:npts);

end

