function BCseis_process_driver
% BCseis_process_driver - program to setup and run BCseis_process for
% applying block manipulation of a signal's CWT as determined from a
% session with BCseis.
%
% Determine appropriate processes and parameters using the BCseis GUI.
% Then set the wanted parameters here to process many seismograms as, for
% example, from an array.

% C. A. Langston, July 9, 2018
% version 1.2, March 1, 2021

%****** params data structure *********************************************
% params.wavelet_type = "morlet" / "shannon" / "mhat" / "hhat"
% params.nvoices      = choice of (4 8 16 32)
% params.outfilesuf   = suffix that will be appended to end of filename for
%                        processed signal.
% params.subdir       = name of subdirectory for resulting captured signal
%                        if blank (''), then the file is written in the
%                        same directory as the input data.
%
% params.nbpblck   = 0, no band block or pass applied
%                  = 1 band block applied
%                  = -1 band pass applied
% params.scale_min = minimum scale for band block/pass
% params.scale_max = maximum scale for band block/pass
% params.bthresh   = percent amplitude for the block
%
% params.nnbnd = 0, no noise model considered
%              = 1, compute a noise model
% params.tstrn = noise start time in seismogram
% params.tfinn = noise finish time in seismogram
%
% params.noisethresh = 0, do not apply non-linear threshold to the noise
%                    = 1, apply hard threshold
%                    = 2, apply soft threshold
%
% params.signalthresh = 0, do not apply non-linear threshold to the signal
%                     = 1, apply hard threshold
%                     = 2, apply soft threshold
%
% params.nsig = number of standard deviations for the block threshold
%             = -1, apply Donoho's Threshold criterion
%             = -2, apply Empirical CDF method to determine threshold
%                  (recommended)
%
% params.nsnr   = 0, do not apply the SNR detection method;
%               = 1, apply.  If applied, it will be completed before hard
%                  thresholding.
% params.nsnrlb = percent lower bound for SNR detection method
%
% params.nextract    = 0, do not extract a CWT block from a signal
%                    = 1, extract a CWT block from an array signal
%                    = 2, delete a CWT block from an array signal
% params.timetol      = time tolerance wanted for shifting the reference
%                      signal relative to the signal. This speeds up the
%                      correlation computation for windowing a block out of
%                      the CWT.  Shifts will occur for +/- params.timtol
% params.extpolyfile = filename for TFR polygon for the reference signal
% params.exttfrfile  = filename for the TFR of the reference signal
% 
%**************************************************************************
% Set the params data structure for this run
% Basic parameters
params.wavelet_type = "morlet";
params.nvoices=16;
params.outfilesuf='st1';
params.subdir='DPZ/';

% Band  pass/block parameters
params.nbpblck = 1;
params.scale_min = 1;
params.scale_max = 1000.0;
params.bthresh=0.0;

% Noise Determination parameters
params.nnbnd = 1;
params.tstrn = 0.0;
params.tfinn = 60.0;

% Noise and/or Signal Threshold parameters
params.noisethresh = 2;
params.signalthresh = 0;
params.nsig = -2.0;

% Signal to Noise function parameters
params.nsnr = 0;
params.nsnrlb = 1.0;

% Array scale-time gating parameters
params.nextract=0;
params.timetol=3.0;
params.extpolyfile='PL-polygon.mat';
params.exttfrfile='PL-CWT-3016.mat';

%**************************************************************************

% set the filename suffix and directory name for this run
fsuffix=params.outfilesuf;
subdirname=params.subdir;

% Read a file of SAC filenames
%   Call uigetfile menu
[filename,dirname]=uigetfile('*.*','Pick a File of SAC filenames for processing');
if filename == 0; return; end

sacnamefile=strcat(dirname,filename);

%   Read in sacfiles from sacnamefile
fid=fopen(sacnamefile,'r');

j=1; while feof(fid)==0
    filedum=[];
    filedum2=[];
    filedum=fgetl(fid);
    [filedum2,fcount,errmsg,nextindex]=sscanf(filedum,'%s',80);
    file(j,1:nextindex-1)=filedum2(1:nextindex-1);
    j=j+1;
    end;  
[nfiles,dull]=size(file);
fclose(fid);

%   Use DOE MatSeis readsac, 'sacdata' is a structure with all the data.
endian='l';
sacdata_in=readsac2( file, endian );

% initialize sacdata_out
sacdata_out=sacdata_in;

%  This next section is performed if the array scale-time gate is being 
%  used.
xv=zeros;
yv=zeros;
W=zeros;
if params.nextract > 0
    %  Read in the polygon file (xv,yv)
    load(params.extpolyfile);
    %  Read in the reference signal TFR (W)
    load(params.exttfrfile);
end
    

% process each file
h=waitbar(0,'Processing Files');
for k=1:nfiles
    xv1=xv;
    yv1=yv;
    Wx_tem=W;
    sacdata_test=BCseis_process(sacdata_in(k),params,Wx_tem,xv1,yv1);
    
    %*******************************************************
    % add the suffix to the filename for later writing
    filename_old=sacdata_out(k).filename;
    
    % parse filename for directory separators
    nc=length(filename_old);
    %  Find all of the / symbols
    indx=0;         % number of underscores in the filename
    ipdx(1:nc)=0.0; % array of indices of /
    for kk=1:nc
        if strcmp(filename_old(kk),'/')
            indx=indx+1;
            ipdx(indx)=kk;
        end
    end
    %*******************************************************
    
    sacdata_out(k)=sacdata_test(1);
    if indx > 0
        sacdata_out(k).filename=strcat(filename_old(1:ipdx(indx)), ...
            subdirname,filename_old(ipdx(indx)+1:nc),'.',fsuffix);
    else
        sacdata_out(k).filename=strcat(subdirname,filename_old,'.',fsuffix);
    end
    
    fprintf('sac file name for k = %i, %s\n',k,sacdata_out(k).filename);
    
    sacdata_test(1).filename=sacdata_out(k).filename;
    % write out the latest file
    ndum=1;
    writesac_a(ndum,sacdata_test);
    
    waitbar(k / nfiles);
end

close(h);

% % write out all files
% writesac_a(nfiles,sacdata_out);

% All Done

end

